package ga.view.streaming.showroom;

import ga.view.interfaces.IPhenotypeSpace;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme3.asset.AssetManager;
import com.jme3.asset.AssetNotFoundException;
import com.jme3.material.Material;
import com.jme3.material.RenderState.FaceCullMode;
import com.jme3.math.ColorRGBA;
import com.jme3.renderer.queue.RenderQueue.ShadowMode;
import com.jme3.scene.Node;
import com.jme3.scene.Spatial;
import com.jme3.system.AppSettings;

/**
 * This is a show room that realizes an {@link IPhenotypeSpace}. It has 4 walls
 * and a floor. The walls are transparent when viewed from outside.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public abstract class ShowRoom extends Node implements IPhenotypeSpace {

  public static final ColorRGBA DEFAULT_COLOR = new ColorRGBA(.5f, .5f, .5f, 1f);

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(ShowRoom.class
      .getName());

  protected Material whiteMaterial;
  protected Material wallMaterial;
  protected Material floorMaterial;
  protected Material ceilingMaterial;

  protected final ShowRoomSettings showRoomSettings;

  private Spatial phenoType;

  /**
   * Instantiates a new show room.
   * 
   * @param name
   *          the name
   * @param assetManager
   *          the asset manager
   * @param settings
   *          the settings
   * @param showRoomSettings
   *          the show room settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ShowRoom(final String name, final AssetManager assetManager,
      final AppSettings settings, final ShowRoomSettings showRoomSettings) {
    super(name);
    this.showRoomSettings = showRoomSettings;

    initMaterials(assetManager, settings);
  }

  /**
   * Gets the wall material.
   * 
   * @return the wall material
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public Material getWallMaterial() {
    return wallMaterial;
  }

  /**
   * Inits the default materials.
   * 
   * @param assetManager
   *          the asset manager
   * @param settings
   *          the app settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void initMaterials(final AssetManager assetManager,
      final AppSettings settings) {

    // init standard white
    whiteMaterial = new Material(assetManager,
        "Common/MatDefs/Light/Lighting.j3md");
    // //// HQ SETTINGS
    if (settings.getBoolean("UseHQMaterial")) {
      whiteMaterial.setFloat("Shininess", 1f);
      whiteMaterial.setBoolean("UseMaterialColors", true);
      whiteMaterial.setColor("Ambient", new ColorRGBA(0.5f, 0.5f, 0.5f, 0.5f));
      whiteMaterial.setColor("Diffuse", DEFAULT_COLOR);
      whiteMaterial.setColor("Specular", ColorRGBA.Black);
      whiteMaterial.setReceivesShadows(true);

      try {
        whiteMaterial.setTexture("ColorRamp",
            assetManager.loadTexture("ga/view/resource/cloudy.png"));
      } catch (final AssetNotFoundException e) {
        LOGGER.log(Level.WARNING, "Texture not found", e);
      }

      try {
        whiteMaterial.setTexture("NormalMap",
            assetManager.loadTexture("furny/textures/Bump_Wallpaper.jpg"));
      } catch (final AssetNotFoundException e) {
        LOGGER.log(Level.WARNING, "Texture not found", e);
      }

      whiteMaterial.setBoolean("Minnaert", false);
      whiteMaterial.setBoolean("WardIso", false);
      whiteMaterial.setBoolean("VTangent", true);
      whiteMaterial.setBoolean("VertexLighting", false);
      whiteMaterial.setBoolean("LowQuality", false);
      whiteMaterial.setBoolean("HighQuality", true);
    } else {
      // /// TODO LAPTOP SETTINGS
      whiteMaterial.setFloat("Shininess", 1f);
      whiteMaterial.setBoolean("UseMaterialColors", true);
      whiteMaterial.setColor("Ambient", DEFAULT_COLOR.mult(0.8f));
      whiteMaterial.setColor("Diffuse", DEFAULT_COLOR);
      whiteMaterial.setColor("Specular", ColorRGBA.Black);

      try {
        whiteMaterial.setTexture("NormalMap",
            assetManager.loadTexture("furny/textures/Bump_Wallpaper.jpg"));
      } catch (final AssetNotFoundException e) {
        LOGGER.log(Level.WARNING, "Texture not found", e);
      }

      whiteMaterial.setReceivesShadows(true);

      whiteMaterial.setBoolean("Minnaert", false);
      whiteMaterial.setBoolean("WardIso", false);
      whiteMaterial.setBoolean("VTangent", false);
      whiteMaterial.setBoolean("VertexLighting", false);
      whiteMaterial.setBoolean("LowQuality", true);
      whiteMaterial.setBoolean("HighQuality", false);
      // / TODO END LAPTOP SETTINGS
    }

    Object o = showRoomSettings.get(ShowRoomSettings.CEILING_MATERIAL);
    if (o != null) {
      if (o instanceof Material) {
        ceilingMaterial = (Material) o;
      } else if (o instanceof String) {
        try {
          ceilingMaterial = assetManager.loadMaterial((String) o);
        } catch (final Exception e) {
          e.printStackTrace();
        }
      }
    }
    if (ceilingMaterial == null) {
      ceilingMaterial = whiteMaterial;
    }

    o = showRoomSettings.get(ShowRoomSettings.WALL_MATERIAL);
    if (o != null) {
      if (o instanceof Material) {
        wallMaterial = (Material) o;
      } else if (o instanceof String) {
        try {
          wallMaterial = assetManager.loadMaterial((String) o);
        } catch (final Exception e) {
          e.printStackTrace();
        }
      }
    }
    if (wallMaterial == null) {
      wallMaterial = whiteMaterial;
    }
    wallMaterial.getAdditionalRenderState().setFaceCullMode(FaceCullMode.Back);

    o = showRoomSettings.get(ShowRoomSettings.FLOOR_MATERIAL);
    if (o != null) {
      if (o instanceof Material) {
        floorMaterial = (Material) o;
      } else if (o instanceof String) {
        try {
          floorMaterial = assetManager.loadMaterial((String) o);
        } catch (final Exception e) {
          e.printStackTrace();
        }
      }
    }
    if (floorMaterial == null) {
      floorMaterial = whiteMaterial;
    }

  }

  /**
   * Sets the phenotype to display inside the room.
   * 
   * @param phenotype
   *          the new phenotype to display.
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setPhenotype(final Spatial phenotype) {
    if (phenoType != null) {
      detachChild(phenoType);
    }

    this.phenoType = phenotype;
    attachChild(phenoType);
    phenoType.setShadowMode(ShadowMode.CastAndReceive);

    updateGeometricState();
  }

  /**
   * Gets the show room settings.
   * 
   * @return the show room settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ShowRoomSettings getShowRoomSettings() {
    return showRoomSettings;
  }

}
